'use strict';

/* --------------------------------------------------------------
 admin_search.js 2018-09-12
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2017 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

/**
 * ## Admin Search Extension
 *
 * Extension for search in orders, customers and categories in the admin panel
 *
 * @module Admin/Extension/admin_search
 * @requires jQueryUI
 * @ignore
 */
gx.extensions.module('admin_search', ['user_configuration_service', 'url_arguments', 'loading_spinner'], function (data) {

    'use strict';

    // ------------------------------------------------------------------------
    // VARIABLE DEFINITION
    // ------------------------------------------------------------------------

    // Elements.

    var $this = $(this),
        $button = $(data.button),
        $dropdown = $('ul.searchable:first'),
        recentSearch = $.trim(decodeURIComponent(jse.libs.url_arguments.getUrlParameters(location.href).search || ''));

    // Current search area.
    var searchArea;

    // Text labels.
    var labels = {
        searchIn: jse.core.lang.translate('admin_search_in_label', 'admin_labels'),
        searchInAlternative: jse.core.lang.translate('admin_search_in_label_alternative', 'admin_labels'),
        orders: jse.core.lang.translate('admin_search_orders', 'admin_labels'),
        invoices: jse.core.lang.translate('admin_search_invoices', 'admin_labels'),
        customers: jse.core.lang.translate('admin_search_customers', 'admin_labels'),
        categories: jse.core.lang.translate('admin_search_categories', 'admin_labels'),
        manual: jse.core.lang.translate('admin_search_manual', 'admin_labels'),
        forum: jse.core.lang.translate('admin_search_forum', 'admin_labels')
    };

    // Key code map.
    var keyMap = {
        ESC: 27,
        ARROW_UP: 38,
        ARROW_DOWN: 40,
        ENTER: 13
    };

    // Library access shortcuts.
    var userConfigurationService = jse.libs.user_configuration_service,
        urlArguments = jse.libs.url_arguments;

    // Configuration settings for UserConfigurationService.
    var configurationContainer = {
        userId: data.customer_id,
        configurationKey: 'recent_search_area'
    };

    // Module object (JSEngine).
    var module = {};

    // ------------------------------------------------------------------------
    // METHODS
    // ------------------------------------------------------------------------

    /**
     * Refreshes the search area variable
     *
     * Shows the new search area in the button
     * @private
     */
    var _refreshSearchArea = function _refreshSearchArea() {
        // Abort if no new search area is provided
        if (!$('.search-item.active').length) {
            console.error('No active list item!');
        }

        // Assign new search area
        searchArea = $('.search-item.active').data('searchArea');
        $this.trigger('refresh:search-area');
    };

    // ------------------------------------------------------------------------
    // INITIALIZATION
    // ------------------------------------------------------------------------

    var _initializeInput = function _initializeInput() {

        // Click event
        $this.parent('*').on('click', function () {
            $this.trigger('refresh:search-area');
            if ($this.val() === '') {
                $this.val(recentSearch);
            }
            $dropdown.trigger('show:dropdown');
            $this.trigger('focus');
        });

        // Keyboard events
        $this.on('keyup', function (event) {
            switch (event.which) {

                // Perform search if enter key is pressed
                case keyMap.ENTER:
                    $this.trigger('perform:search');
                    break;

                // Close dropdown if escape key is pressed
                case keyMap.ESC:
                    $dropdown.trigger('hide:dropdown');
                    return;

                // Navigate up in dropdown
                case keyMap.ARROW_UP:
                    $dropdown.trigger('select:item:previous');
                    break;
                case keyMap.ARROW_DOWN:
                    $dropdown.trigger('select:item:next');
                    break;
            }
            $dropdown.trigger('refresh:search-item');
        });

        // Search events
        $this.on('perform:search', function () {
            var inputValue = encodeURIComponent($this.val()),
                openMode = '_self',
                url;

            switch (searchArea) {
                case 'customers':
                    url = ['customers.php', '?search=', inputValue].join('');
                    break;
                case 'categories':
                    url = ['categories.php', '?search=', inputValue].join('');
                    break;
                case 'orders':
                    url = ['admin.php', '?', $.param({
                        do: 'OrdersOverview',
                        filter: {
                            number: inputValue
                        }
                    })].join('');
                    break;
                case 'invoices':
                    url = ['admin.php', '?', $.param({
                        do: 'InvoicesOverview',
                        filter: {
                            invoiceNumber: inputValue
                        }
                    })].join('');
                    break;
                case 'manual':
                    url = ['admin.php', '?', $.param({
                        do: 'DirectHelpProxy/GoToManual',
                        search: inputValue
                    })].join('');
                    openMode = '_blank';
                    $dropdown.trigger('hide:dropdown');
                    break;
                case 'forum':
                    url = ['admin.php', '?', $.param({
                        do: 'DirectHelpProxy/GoToForum',
                        number: Math.floor(Math.random() * 99999999 + 1),
                        search: inputValue
                    })].join('');
                    openMode = '_blank';
                    $dropdown.trigger('hide:dropdown');
                    break;
            }

            // Display loading spinner.
            var $spinner = jse.libs.loading_spinner.show($dropdown, '9999');

            userConfigurationService.set({
                data: $.extend(configurationContainer, {
                    configurationValue: searchArea
                }),
                onSuccess: function onSuccess() {
                    window.open(url, openMode);
                    jse.libs.loading_spinner.hide($spinner);
                },
                onError: function onError() {
                    window.open(url, openMode);
                    jse.libs.loading_spinner.hide($spinner);
                }
            });
        });

        // Remove placeholder when input is inactive
        $this.on('blur', function () {
            $dropdown.trigger('hide:dropdown');
        });
    };

    var _initializeButton = function _initializeButton() {
        $button.on('click', function () {
            $this.trigger('refresh:search-area');
            $this.val(recentSearch);
            $dropdown.trigger('show:dropdown');
            $this.trigger('focus');
        });
    };

    var _initializeDropdown = function _initializeDropdown() {
        // Select item
        $dropdown.on('select:item', function () {
            $dropdown.find('li[data-search-area=' + searchArea + ']').addClass('active');
        });

        // Show event
        $dropdown.on('show:dropdown', function () {
            $dropdown.fadeIn();
            $dropdown.trigger('select:item');
            $dropdown.trigger('refresh:search-item');
        });

        // Select first item
        $dropdown.on('select:item:first', function () {
            var $activeListItem = $dropdown.find('li.search-item.active');
            var $firstListItem = $dropdown.find('li.search-item:first');
            $activeListItem.removeClass('active');
            $firstListItem.addClass('active');
            _refreshSearchArea();
            $dropdown.trigger('select:item');
        });

        $dropdown.on('select:item:last', function () {
            var $activeListItem = $dropdown.find('li.search-item.active');
            var $lastListItem = $dropdown.find('li.search-item:last');
            $activeListItem.removeClass('active');
            $lastListItem.addClass('active');
            _refreshSearchArea();
            $dropdown.trigger('select:item');
        });

        // Select previous item event
        $dropdown.on('select:item:previous', function () {
            var $activeListItem = $dropdown.find('li.search-item.active');
            var $prev = $activeListItem.prev();

            if ($prev.length) {
                $activeListItem.removeClass('active');
                $prev.addClass('active');
                _refreshSearchArea();
                $dropdown.trigger('select:item');
            } else {
                $dropdown.trigger('select:item:last');
            }
        });

        // Select previous item event
        $dropdown.on('select:item:next', function () {
            var $activeListItem = $dropdown.find('li.search-item.active');
            var $next = $activeListItem.next();

            if ($next.length) {
                $activeListItem.removeClass('active');
                $next.addClass('active');
                _refreshSearchArea();
                $dropdown.trigger('select:item');
            } else {
                $dropdown.trigger('select:item:first');
            }
        });

        // Hide event
        $dropdown.on('hide:dropdown', function () {
            $dropdown.fadeOut();
        });

        // Item click event
        $dropdown.on('click', function (event) {
            event.stopPropagation();

            $dropdown.find('li').removeClass('active');

            var $elementToActivate = $(event.target).is('span') ? $(event.target).parents('li:first') : $(event.target);

            $elementToActivate.addClass('active');

            _refreshSearchArea();
            $dropdown.trigger('hide:dropdown');
            $this.trigger('perform:search');
        });

        // Item search event
        $dropdown.on('refresh:search-item', function () {
            $('.search-item').each(function () {
                // Update search query
                $(this).find('.search-query-item').text($this.val());

                // Update search description
                var searchAreaText = [labels.searchIn, labels[$(this).data('searchArea')]].join(' ');

                if ($(this).attr('data-search-area') === 'manual' || $(this).attr('data-search-area') === 'forum') {
                    searchAreaText = [labels.searchInAlternative, labels[$(this).data('searchArea')]].join(' ');
                }

                $(this).find('.search-query-description').text(searchAreaText);
            });
        });
    };

    var _initializeRecentSearch = function _initializeRecentSearch() {
        $(document).on('JSENGINE_INIT_FINISHED', function () {
            if (recentSearch != '') {
                $this.prop('value', recentSearch);
                $this.focus();
                $dropdown.trigger('show:dropdown');
                $dropdown.trigger('refresh:search-item');
            }
        });
    };

    /**
     * Initialize method of the extension, called by the engine.
     */
    module.init = function (done) {
        _initializeInput();
        _initializeDropdown();
        _initializeButton();
        _initializeRecentSearch();

        searchArea = data.recentSearchArea || 'categories';
        $dropdown.trigger('select:item');

        done();
    };

    // Return data to module engine.
    return module;
});
//# sourceMappingURL=data:application/json;charset=utf8;base64,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
